#!/usr/bin/perl -w

# Copyright 2003, Jonathan R. Finger (jonathan_finger@hotmail.com).
# All rights reserved.
# This library is free software; you can redistribute it and/or modify
# it under the terms of the GNU Public Liscence (GPL) version 2
# or later.

package date_arithmetic;
require Exporter;
@ISA = qw(Exporter);
@EXPORT_OK = qw(convert_date_to_number convert_number_to_date validate_date
                month_name day_name system_time_to_number);

use strict;
use constant SECONDS_IN_DAY  => 86400;

my $date_fmt = "%s/%02d/%02d";

# Conversion formulas are from
#   Calenderical Caluculation
#   by Dershowitz and Reingold
#   ISBN: 0-521-56474-3

sub is_leap_year
{
  my $year = shift;
  if (($year % 4) == 0)
    {
      my $i = $year % 400;
      if (($i != 100) && ($i != 200) && ($i != 300))
       {
         return(1);
       }
    }
  return(0);
}

sub convert_date_to_number
{
  my ($year, $month, $day) = @_;
  $year--; # to make formula simpler
  my $day_number = ($year * 365) + int($year/4) - int($year/100) + int($year/400)
      + int(((367 * $month) - 362)/12) + $day;
  if ($month >2) 
    { 
      $day_number -= 2;
      if (is_leap_year($year + 1))
        {
          $day_number++;
        }
    }
  return($day_number);
}



sub  convert_number_to_date
{
  my $day_number = shift() - 1; # -1 to simply formulas
  my $d1 = $day_number % 146097;
  my $d2 = $d1 % 36524;
  my $n100 = int($d1/36524);
  my $d3 = $d2 % 1461;
  my $n1 = int($d3/365);
  my $year =   (400 * int($day_number/146097)) + (100 * $n100)
                + (4 * int ($d2/1461)) + $n1;
  if (($n1 != 4) && ($n100 != 4))
    {
      $year++;
    }
  $day_number++;
  $year--; # to make next formula simpler
  my $first_day_of_year =    int($year * 365) + int($year/4) 
                           - int($year/100) + int($year/400) + 1; 
                          # convert_date_to_number($year, 1, 1);
  $year++; # undo previous decrement
  my $leap_year = 0;
  if (is_leap_year($year)) { $leap_year = 1 }
  my $first_of_March = $first_day_of_year + 59 + $leap_year;
  my $correction = 0;
  if ($day_number >= $first_of_March)
    {
      $correction = 2 - $leap_year;
    }
  my $month = int(((12 * ($day_number - $first_day_of_year + $correction)) + 373)/367);
  my $day = $day_number - $first_day_of_year - int(((367 * $month) - 362)/12) + 1;
  if ($month >2) 
    { 
      $day += 2 - $leap_year;
    }
#           $day_number - convert_date_to_number($year, $month, 1) + 1
  return($year, $month, $day);
# +1 for month and day since first day or month is 1, not 0
}


sub month_name
{
  my $month = shift;
  return (("January", "February", "March", "April", "May", "June", "July",
          "August", "September", "October", "November", "December")[$month - 1]);
}

sub day_name
{
  return(("Sunday", "Monday", "Tuesday", "Wednesday", 
	  "Thursday", "Friday",
	  "Saturday")[convert_date_to_number(@_) % 7]);
}

#
#  Usage validate_date(year, month, day)
#   returns TRUE if date is valid
#           FALST if date is invalid
sub validate_date
{
    my ($year, $month, $day) = @_;
    if ("$year$month$day" !~ /^\d\d*$/) { return(0) }
  my ($c_year, $c_month, $c_day)
      = convert_number_to_date(convert_date_to_number($year, $month, $day));
  return (($c_year == $year) && ($c_month == $month) && ($c_day == $day));
}

sub system_time_to_number
{
# takes system time (seconds from OS dependent origin and converts
# to the number of days since 1/1/1
  my $time = shift;
  my @data = (localtime($time))[5,4,3];
  return(convert_date_to_number($data[0] + 1900, $data[1] + 1, 
				$data[2]));
}

1;





