/*

 (c) 2002 Brian Kelly under the terms of the GNU GPL
	http://www.gnu.org/copyleft/gpl.html

*/

/*
	a command-line check-book register program
		well, doesn't have to be command-line only, but it's starting that way
		want to store transactions in plain text, human readable files
		as designed, you should even be able to go back, hand edit files, and nothing will be damaged
			gets more complicated when we have different files for each month

	interface:
		- Simply execute the command with arguments specifying the transaction
		- Suggestions so far:
			debit: remove money, as in writing a check, charging account, etc.
				<exec> -d 112.43 Paycheck
			credit: add money, as in deposits
				<exec> -c 45.18 "November rent"
		- So, that's <exec> followed by options
		- both the -c and -d options take two arguments:
			* amount of transaction, interpretted with two decimal points precision, by default
			* a comment about the transaction, bounded by quotes if need be
		- There will also be an option to simply print out the current balance

	So, we need the following facilities:
		- Ability to parse through command line options, and make sense of poorly formed options, to a degree
			(invert operation on negatives? append or chop off decimals as needed?)
		- Ability to write out (usually just append) to files to store transactions for later use
		- Ability to read in and calculate current total, based on new, if any, transactions
		- Store all information in hidden directory in user's home dir (i.e. ~/.banking-program/<data is here>)

	Possible additional features:
		- Support more than one account (so specify account on command-line? default account?)
		- Cancel transactions (maybe just do it manually by adding inverse transaction to list)

	Ideas as they arise..
		- to input decimals as numbers, scan through, add and multiply total, then divide by 100
		- to calculate current total, put initial balance at top of file, then run through transactions
			store debits as negative numbers, credits as positive
		- store each month in a seperate file; when creating new file, carry over last's balance as this's initial balance
		- thinking the file will have the following format, in BNF, but left justified
			<initial-balance>
			(#<date>
			<transaction amount>)*
		- said format allows us to get an easy initial to work from, and easy identifier ('#') so we know to skip that line in calculating the balance

	So, programming-wise,
		- Classes? Don't wanna..
		- More C like, but with C++ IOStreams for input/output and file handling
		- Do separate files for major functions, called in by main()

*/


#include <iostream.h>
#include <fstream.h>
#include "trans.h"

void usage();
void interactive();

int main( int argc, char * argv[] )
{
	//Hmm, let's have main parse the command line options, and call external functions to do the work, based on those options

	bool args = false; // Have we found there to be some arguments

	for ( int i = 1; i <= (argc - 1); i++ )
	{
		args = true;

		if ( argv[i][0] == '-' )
		{
			switch ( argv[i][1] )
			{
				case 'v':
				{
					cout << "brinance version 1.1" << endl;
					break;
				}
				case 'h':
				{
					usage();
					break;
				}
				case 'd':
				{
					if ( argc < i+3 ){
						cout << "Need an amount to debit and a comment" << endl;
						usage();
					}
					else {
						debit( argv[i+1], argv[i+2] );
						i += 2; //Incremented i to skip next argument
						balance();
					}
					break;
				}
				case 'c':
				{
					if ( argc < i+3 ){
						cout << "Need an amount to credit and a comment" << endl;
						usage();
					}
					else {
						credit( argv[i+1], argv[i+2] ); //Incremented i to skip next argument
						i += 2;
						balance();
					}
					break;
				}
				case 'b':
				{
					balance();
					break;
				}
				case '-':
				{
					if ( strcmp( argv[i]+2, "help" ) == 0 ) {
						usage();
					}
					else if ( strcmp( argv[i]+2, "version" ) == 0 ) {
						cout << "brinance version 1.1" << endl;
					}
					else if ( strcmp( argv[i]+2, "credit" ) == 0 ) {
						if ( argc < i+3 ){
							cout << "Need an amount to credit" << endl;
							usage();
						}
						else {
							credit( argv[i+1], argv[i+2] ); //Incremented i to skip next argument
							i += 2;
						}
					}
					else if ( strcmp( argv[i]+2, "debit" ) == 0 ) {
						if ( argc < i+3 ){
							cout << "Need an amount to debit" << endl;
							usage();
						}
						else {
							debit( argv[i+1], argv[i+2] ); //Incremented i to skip next argument
							i += 2;
						}
					}
					else if ( strcmp( argv[i]+2, "balance" ) == 0 ) {
						balance();
					}

					else
					{
						cout << "unrecognized option: --" << argv[i]+2 << endl;
						usage();
					}
					
					break;
				}
				default:
				{
					cout << "Unrecognized option: -" << argv[i][1] << endl;
					usage();
				}
			}
		}
		else {
			usage();
		}
	}

	if ( !args ) // Are there not args?
		interactive(); //go into interactive mode

	return 0;
}

void usage()
{
	cout << "Usage:\n";

	cout << "\tbrinance <options>\n\n";

	cout << "Options:\n";

	cout << "\t-d <amount> <comment>\tcreates a debit of said amount commented\n\t--debit <amount> <comment>\n\n";
	cout << "\t-c <amount> <comment>\tcreates a credit of said amount commented\n\t--credit <amount> <comment>\n\n";
	cout << "\t-b\t\t\tprint the current balance\n\t--balance\n\n";
	cout << "\t-h\t\t\tprint this help message\n\t--help\n\n";
	cout << "\t-v\t\t\tprint version message\n\t--version";

	cout << endl;
	exit(0);
}

void interactive()
{
	char choice = 'a';
	char inputA[20]; // input amount
	char inputC[80]; // input comment

	cout << "Brinance interactive mode\n" << endl;

	while ( choice != '0' )
	{
		cout << "Please enter the number of your choice.\n" << endl;

		cout << "1) Debit\n";
		cout << "2) Credit\n";
		cout << "3) Current Balance\n";

		cout << "\n0) Quit\n\n> ";

		cin >> choice;

		switch ( choice )
		{
			case '1': {
				cout << "Enter amount to debit\n> ";
				cin >> inputA;

				cout << "Enter comment for this debit\n> ";
				cin.getline( inputC, 81 );

				debit( inputA, inputC );

				break;
			}
			case '2': {
				cout << "Enter amount to credit\n> ";
				cin >> inputA;

				cout << "Enter comment for this credit\n> ";
				cin.getline( inputC, 81 );

				credit( inputA, inputC );

				break;
			}
			case '3': {
				balance();
				break;
			}
			case '0': {
				break;
			}
			default: {
				cout << "Invalid choice: " << choice << endl;
				break;
			}
		}

		cout << endl;
	}

	cout << "Bye!" << endl;
}

